<?php
/**
 * API - Cotizaciones Públicas
 * Endpoint para recibir solicitudes del cotizador web
 * 
 * POST /api/cotizacion-publica.php
 */

// Configuración de errores (desactivar en producción)
error_reporting(E_ALL);
ini_set('display_errors', 0);

// Headers CORS y JSON
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Manejar preflight request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Solo aceptar POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método no permitido']);
    exit;
}

// =====================================================
// CONFIGURACIÓN DE BASE DE DATOS
// =====================================================
$config = [
    'host' => 'localhost',
    'dbname' => 'ataralcl_export',  // ← Cambiar por tu base de datos
    'user' => 'root',               // ← Cambiar por tu usuario
    'pass' => '',                   // ← Cambiar por tu contraseña
    'charset' => 'utf8mb4'
];

// =====================================================
// CONFIGURACIÓN DE EMAIL (opcional)
// =====================================================
$emailConfig = [
    'enabled' => true,
    'to' => 'ventas@aflcargo.net',  // ← Email donde llegarán las notificaciones
    'from' => 'noreply@aflcargo.net',
    'fromName' => 'AFL Cargo - Cotizador Web'
];

// =====================================================
// FUNCIONES
// =====================================================

function conectarDB($config) {
    try {
        $dsn = "mysql:host={$config['host']};dbname={$config['dbname']};charset={$config['charset']}";
        $pdo = new PDO($dsn, $config['user'], $config['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES => false
        ]);
        return $pdo;
    } catch (PDOException $e) {
        throw new Exception('Error de conexión a base de datos');
    }
}

function validarDatos($data) {
    $errores = [];
    
    // Campos requeridos
    $requeridos = ['tipoTransporte', 'origen', 'destino', 'categoria', 'peso', 'nombre', 'email', 'telefono'];
    foreach ($requeridos as $campo) {
        if (empty($data[$campo])) {
            $errores[] = "El campo '$campo' es requerido";
        }
    }
    
    // Validar email
    if (!empty($data['email']) && !filter_var($data['email'], FILTER_VALIDATE_EMAIL)) {
        $errores[] = "El email no es válido";
    }
    
    // Validar tipo de transporte
    $tiposValidos = ['aereo', 'maritimo', 'terrestre'];
    if (!empty($data['tipoTransporte']) && !in_array($data['tipoTransporte'], $tiposValidos)) {
        $errores[] = "Tipo de transporte no válido";
    }
    
    return $errores;
}

function sanitizar($valor) {
    if (is_string($valor)) {
        return htmlspecialchars(strip_tags(trim($valor)), ENT_QUOTES, 'UTF-8');
    }
    return $valor;
}

function enviarEmailNotificacion($data, $emailConfig) {
    if (!$emailConfig['enabled']) return;
    
    $tipoTransporte = [
        'aereo' => '✈️ Aéreo',
        'maritimo' => '🚢 Marítimo',
        'terrestre' => '🚛 Terrestre'
    ];
    
    $asunto = "Nueva solicitud de cotización - {$data['nombre']}";
    
    $mensaje = "
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #1e3a5f, #0c4a6e); color: white; padding: 20px; border-radius: 10px 10px 0 0; }
            .content { background: #f5f5f5; padding: 20px; }
            .field { margin-bottom: 15px; }
            .label { font-weight: bold; color: #666; font-size: 12px; text-transform: uppercase; }
            .value { font-size: 16px; margin-top: 4px; }
            .estimate { background: #d4edda; padding: 15px; border-radius: 8px; text-align: center; margin: 20px 0; }
            .footer { background: #333; color: white; padding: 15px; text-align: center; border-radius: 0 0 10px 10px; font-size: 12px; }
            .btn { display: inline-block; background: #8b5cf6; color: white; padding: 12px 24px; text-decoration: none; border-radius: 6px; margin-top: 15px; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h2 style='margin:0;'>🚀 Nueva Solicitud de Cotización</h2>
                <p style='margin:10px 0 0;opacity:0.9;'>Recibida el " . date('d/m/Y H:i') . "</p>
            </div>
            <div class='content'>
                <h3>📦 Datos del Envío</h3>
                <div class='field'>
                    <div class='label'>Tipo de Transporte</div>
                    <div class='value'>" . ($tipoTransporte[$data['tipoTransporte']] ?? $data['tipoTransporte']) . "</div>
                </div>
                <div class='field'>
                    <div class='label'>Ruta</div>
                    <div class='value'>{$data['origen']} → {$data['destino']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Categoría</div>
                    <div class='value'>{$data['categoria']}</div>
                </div>
                <div class='field'>
                    <div class='label'>Peso Aproximado</div>
                    <div class='value'>" . number_format($data['peso'], 0, ',', '.') . " kg</div>
                </div>
                " . (!empty($data['descripcion']) ? "
                <div class='field'>
                    <div class='label'>Descripción</div>
                    <div class='value'>{$data['descripcion']}</div>
                </div>" : "") . "
                
                <div class='estimate'>
                    <div class='label'>Estimación mostrada al cliente</div>
                    <div style='font-size:24px;font-weight:bold;color:#155724;'>
                        \${$data['estimacion']['min']} - \${$data['estimacion']['max']} USD
                    </div>
                </div>
                
                <h3>👤 Datos de Contacto</h3>
                <div class='field'>
                    <div class='label'>Nombre</div>
                    <div class='value'>{$data['nombre']}</div>
                </div>
                " . (!empty($data['empresa']) ? "
                <div class='field'>
                    <div class='label'>Empresa</div>
                    <div class='value'>{$data['empresa']}</div>
                </div>" : "") . "
                <div class='field'>
                    <div class='label'>Email</div>
                    <div class='value'><a href='mailto:{$data['email']}'>{$data['email']}</a></div>
                </div>
                <div class='field'>
                    <div class='label'>Teléfono</div>
                    <div class='value'><a href='tel:{$data['telefono']}'>{$data['telefono']}</a></div>
                </div>
                <div class='field'>
                    <div class='label'>Prefiere contacto por</div>
                    <div class='value'>{$data['preferencia']}</div>
                </div>
                
                <center>
                    <a href='https://wa.me/{$data['telefono']}' class='btn'>💬 Contactar por WhatsApp</a>
                </center>
            </div>
            <div class='footer'>
                AFL Cargo - Sistema de Cotizaciones Web<br>
                Este email fue generado automáticamente
            </div>
        </div>
    </body>
    </html>
    ";
    
    $headers = [
        'MIME-Version: 1.0',
        'Content-type: text/html; charset=UTF-8',
        "From: {$emailConfig['fromName']} <{$emailConfig['from']}>",
        "Reply-To: {$data['email']}"
    ];
    
    @mail($emailConfig['to'], $asunto, $mensaje, implode("\r\n", $headers));
}

// =====================================================
// PROCESAR SOLICITUD
// =====================================================

try {
    // Obtener datos JSON
    $json = file_get_contents('php://input');
    $data = json_decode($json, true);
    
    if (!$data) {
        throw new Exception('Datos JSON inválidos');
    }
    
    // Validar
    $errores = validarDatos($data);
    if (!empty($errores)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'errors' => $errores]);
        exit;
    }
    
    // Conectar a BD
    $pdo = conectarDB($config);
    
    // Preparar datos
    $params = [
        ':tipo_transporte' => sanitizar($data['tipoTransporte']),
        ':origen' => sanitizar($data['origen']),
        ':destino' => sanitizar($data['destino']),
        ':categoria' => sanitizar($data['categoria']),
        ':descripcion' => sanitizar($data['descripcion'] ?? ''),
        ':peso_kg' => floatval($data['peso']),
        ':estimacion_min' => floatval($data['estimacion']['min'] ?? 0),
        ':estimacion_max' => floatval($data['estimacion']['max'] ?? 0),
        ':nombre' => sanitizar($data['nombre']),
        ':empresa' => sanitizar($data['empresa'] ?? ''),
        ':email' => sanitizar($data['email']),
        ':telefono' => sanitizar($data['telefono']),
        ':preferencia_contacto' => sanitizar($data['preferencia'] ?? 'email'),
        ':idioma' => sanitizar($data['idioma'] ?? 'es'),
        ':ip_cliente' => $_SERVER['REMOTE_ADDR'] ?? '',
        ':user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? '', 0, 255),
        ':fuente' => sanitizar($data['fuente'] ?? 'web')
    ];
    
    // Insertar
    $sql = "INSERT INTO cotizaciones_publicas (
                tipo_transporte, origen, destino, categoria, descripcion, peso_kg,
                estimacion_min, estimacion_max, nombre, empresa, email, telefono,
                preferencia_contacto, idioma, ip_cliente, user_agent, fuente
            ) VALUES (
                :tipo_transporte, :origen, :destino, :categoria, :descripcion, :peso_kg,
                :estimacion_min, :estimacion_max, :nombre, :empresa, :email, :telefono,
                :preferencia_contacto, :idioma, :ip_cliente, :user_agent, :fuente
            )";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    $id = $pdo->lastInsertId();
    
    // Enviar notificación por email
    enviarEmailNotificacion($data, $emailConfig);
    
    // Respuesta exitosa
    echo json_encode([
        'success' => true,
        'message' => 'Solicitud recibida correctamente',
        'id' => $id
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error al procesar la solicitud: ' . $e->getMessage()
    ]);
}
