/**
 * AFL Cargo - Widget de Cotización Embebible
 * 
 * INSTALACIÓN:
 * Agrega este código antes del cierre </body> en tu sitio web:
 * 
 * <script src="https://tudominio.com/widget/afl-quote-widget.js"></script>
 * <script>
 *   AFLCargoWidget.init({
 *     whatsapp: '56912345678',     // Tu número de WhatsApp Business
 *     color: '#8b5cf6',            // Color principal (opcional)
 *     position: 'right',           // 'left' o 'right' (opcional)
 *     lang: 'es'                   // 'es' o 'en' (opcional, detecta automáticamente)
 *   });
 * </script>
 * 
 * @version 1.0.0
 */

(function() {
    'use strict';

    // Evitar múltiples inicializaciones
    if (window.AFLCargoWidget && window.AFLCargoWidget.initialized) return;

    const AFLCargoWidget = {
        initialized: false,
        config: {
            whatsapp: '56912345678',
            color: '#8b5cf6',
            position: 'right',
            lang: navigator.language.substring(0, 2) === 'en' ? 'en' : 'es',
            baseUrl: '' // Se puede configurar si el widget está en otro dominio
        },

        translations: {
            es: {
                button_text: 'Cotizar Envío',
                title: '¿Necesitas enviar carga?',
                subtitle: 'Cotiza en 2 minutos',
                step1: '¿Qué tipo de transporte?',
                air: 'Aéreo', sea: 'Marítimo', land: 'Terrestre',
                step2: 'Ruta de envío',
                origin: 'Origen', destination: 'Destino',
                step3: 'Peso aproximado',
                step4: 'Tus datos',
                name: 'Nombre', email: 'Email', phone: 'Teléfono',
                submit: 'Solicitar Cotización',
                success: '¡Gracias! Te contactaremos pronto.',
                whatsapp: 'O escríbenos por WhatsApp',
                close: 'Cerrar',
                sending: 'Enviando...'
            },
            en: {
                button_text: 'Get Quote',
                title: 'Need to ship cargo?',
                subtitle: 'Quote in 2 minutes',
                step1: 'Transport type?',
                air: 'Air', sea: 'Sea', land: 'Ground',
                step2: 'Shipping route',
                origin: 'Origin', destination: 'Destination',
                step3: 'Approximate weight',
                step4: 'Your info',
                name: 'Name', email: 'Email', phone: 'Phone',
                submit: 'Request Quote',
                success: 'Thanks! We\'ll contact you soon.',
                whatsapp: 'Or message us on WhatsApp',
                close: 'Close',
                sending: 'Sending...'
            }
        },

        init: function(options) {
            if (this.initialized) return;
            
            // Merge options
            Object.assign(this.config, options || {});
            
            // Inject styles
            this.injectStyles();
            
            // Create widget elements
            this.createButton();
            this.createModal();
            
            this.initialized = true;
        },

        t: function(key) {
            return this.translations[this.config.lang][key] || key;
        },

        injectStyles: function() {
            const css = `
                @import url('https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap');
                
                .afl-widget-btn {
                    position: fixed;
                    bottom: 24px;
                    ${this.config.position}: 24px;
                    z-index: 99999;
                    display: flex;
                    align-items: center;
                    gap: 8px;
                    padding: 14px 24px;
                    background: linear-gradient(135deg, ${this.config.color}, #3b82f6);
                    color: white;
                    border: none;
                    border-radius: 50px;
                    font-family: 'Inter', sans-serif;
                    font-size: 15px;
                    font-weight: 600;
                    cursor: pointer;
                    box-shadow: 0 4px 20px rgba(0,0,0,0.25);
                    transition: all 0.3s ease;
                    animation: afl-pulse 2s infinite;
                }
                
                .afl-widget-btn:hover {
                    transform: scale(1.05);
                    box-shadow: 0 6px 30px rgba(0,0,0,0.3);
                }
                
                @keyframes afl-pulse {
                    0%, 100% { box-shadow: 0 4px 20px rgba(139, 92, 246, 0.4); }
                    50% { box-shadow: 0 4px 30px rgba(139, 92, 246, 0.6); }
                }
                
                .afl-widget-btn svg {
                    width: 20px;
                    height: 20px;
                }
                
                .afl-modal-overlay {
                    position: fixed;
                    inset: 0;
                    background: rgba(0,0,0,0.6);
                    backdrop-filter: blur(4px);
                    z-index: 100000;
                    display: none;
                    align-items: center;
                    justify-content: center;
                    padding: 16px;
                    opacity: 0;
                    transition: opacity 0.3s ease;
                }
                
                .afl-modal-overlay.active {
                    display: flex;
                    opacity: 1;
                }
                
                .afl-modal {
                    background: linear-gradient(135deg, #1e293b 0%, #0f172a 100%);
                    border-radius: 24px;
                    width: 100%;
                    max-width: 440px;
                    max-height: 90vh;
                    overflow-y: auto;
                    color: white;
                    font-family: 'Inter', sans-serif;
                    box-shadow: 0 25px 50px rgba(0,0,0,0.5);
                    transform: scale(0.9) translateY(20px);
                    transition: transform 0.3s ease;
                }
                
                .afl-modal-overlay.active .afl-modal {
                    transform: scale(1) translateY(0);
                }
                
                .afl-modal-header {
                    padding: 24px;
                    text-align: center;
                    border-bottom: 1px solid rgba(255,255,255,0.1);
                    position: relative;
                }
                
                .afl-modal-close {
                    position: absolute;
                    top: 16px;
                    right: 16px;
                    background: rgba(255,255,255,0.1);
                    border: none;
                    color: white;
                    width: 32px;
                    height: 32px;
                    border-radius: 50%;
                    cursor: pointer;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    transition: background 0.2s;
                }
                
                .afl-modal-close:hover {
                    background: rgba(255,255,255,0.2);
                }
                
                .afl-modal-title {
                    font-size: 24px;
                    font-weight: 700;
                    margin: 0 0 4px;
                }
                
                .afl-modal-subtitle {
                    color: rgba(255,255,255,0.6);
                    font-size: 14px;
                    margin: 0;
                }
                
                .afl-modal-body {
                    padding: 24px;
                }
                
                .afl-step {
                    display: none;
                }
                
                .afl-step.active {
                    display: block;
                    animation: afl-fadeIn 0.3s ease;
                }
                
                @keyframes afl-fadeIn {
                    from { opacity: 0; transform: translateX(20px); }
                    to { opacity: 1; transform: translateX(0); }
                }
                
                .afl-step-label {
                    font-size: 14px;
                    color: ${this.config.color};
                    margin-bottom: 16px;
                    font-weight: 500;
                }
                
                .afl-transport-grid {
                    display: grid;
                    grid-template-columns: repeat(3, 1fr);
                    gap: 12px;
                }
                
                .afl-transport-btn {
                    background: rgba(255,255,255,0.05);
                    border: 2px solid rgba(255,255,255,0.1);
                    border-radius: 16px;
                    padding: 20px 12px;
                    text-align: center;
                    cursor: pointer;
                    transition: all 0.2s;
                    color: white;
                }
                
                .afl-transport-btn:hover {
                    background: rgba(255,255,255,0.1);
                    border-color: rgba(255,255,255,0.2);
                }
                
                .afl-transport-btn.selected {
                    background: linear-gradient(135deg, rgba(139,92,246,0.3), rgba(59,130,246,0.3));
                    border-color: ${this.config.color};
                }
                
                .afl-transport-btn .icon {
                    font-size: 28px;
                    margin-bottom: 8px;
                }
                
                .afl-transport-btn .label {
                    font-size: 13px;
                    font-weight: 500;
                }
                
                .afl-input-group {
                    margin-bottom: 16px;
                }
                
                .afl-input-label {
                    display: block;
                    font-size: 13px;
                    color: rgba(255,255,255,0.6);
                    margin-bottom: 6px;
                }
                
                .afl-input {
                    width: 100%;
                    padding: 14px 16px;
                    background: rgba(255,255,255,0.05);
                    border: 1px solid rgba(255,255,255,0.1);
                    border-radius: 12px;
                    color: white;
                    font-size: 15px;
                    font-family: inherit;
                    transition: all 0.2s;
                    box-sizing: border-box;
                }
                
                .afl-input:focus {
                    outline: none;
                    border-color: ${this.config.color};
                    box-shadow: 0 0 0 3px rgba(139,92,246,0.2);
                }
                
                .afl-input::placeholder {
                    color: rgba(255,255,255,0.3);
                }
                
                .afl-weight-display {
                    text-align: center;
                    margin-bottom: 16px;
                }
                
                .afl-weight-value {
                    font-size: 48px;
                    font-weight: 700;
                    background: linear-gradient(135deg, #60a5fa, #a78bfa);
                    -webkit-background-clip: text;
                    -webkit-text-fill-color: transparent;
                }
                
                .afl-weight-unit {
                    font-size: 20px;
                    color: rgba(255,255,255,0.5);
                    margin-left: 4px;
                }
                
                .afl-range {
                    width: 100%;
                    height: 8px;
                    -webkit-appearance: none;
                    background: linear-gradient(90deg, #3b82f6, #8b5cf6, #ec4899);
                    border-radius: 4px;
                    outline: none;
                }
                
                .afl-range::-webkit-slider-thumb {
                    -webkit-appearance: none;
                    width: 24px;
                    height: 24px;
                    background: white;
                    border-radius: 50%;
                    cursor: pointer;
                    box-shadow: 0 2px 8px rgba(0,0,0,0.3);
                }
                
                .afl-estimate {
                    background: linear-gradient(135deg, rgba(16,185,129,0.2), rgba(5,150,105,0.2));
                    border: 1px solid rgba(16,185,129,0.3);
                    border-radius: 16px;
                    padding: 16px;
                    text-align: center;
                    margin-top: 20px;
                }
                
                .afl-estimate-label {
                    font-size: 13px;
                    color: #10b981;
                    margin-bottom: 4px;
                }
                
                .afl-estimate-value {
                    font-size: 24px;
                    font-weight: 700;
                }
                
                .afl-btn-row {
                    display: flex;
                    gap: 12px;
                    margin-top: 24px;
                }
                
                .afl-btn {
                    flex: 1;
                    padding: 14px;
                    border-radius: 12px;
                    font-size: 15px;
                    font-weight: 600;
                    cursor: pointer;
                    transition: all 0.2s;
                    font-family: inherit;
                    border: none;
                }
                
                .afl-btn-secondary {
                    background: rgba(255,255,255,0.1);
                    color: white;
                }
                
                .afl-btn-secondary:hover {
                    background: rgba(255,255,255,0.15);
                }
                
                .afl-btn-primary {
                    background: linear-gradient(135deg, ${this.config.color}, #3b82f6);
                    color: white;
                }
                
                .afl-btn-primary:hover {
                    transform: translateY(-2px);
                    box-shadow: 0 4px 12px rgba(139,92,246,0.4);
                }
                
                .afl-btn-primary:disabled {
                    opacity: 0.5;
                    cursor: not-allowed;
                    transform: none;
                }
                
                .afl-btn-whatsapp {
                    background: #25D366;
                    color: white;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    gap: 8px;
                }
                
                .afl-btn-whatsapp:hover {
                    background: #22c55e;
                }
                
                .afl-success {
                    text-align: center;
                    padding: 20px 0;
                }
                
                .afl-success-icon {
                    width: 80px;
                    height: 80px;
                    background: linear-gradient(135deg, #10b981, #059669);
                    border-radius: 50%;
                    display: flex;
                    align-items: center;
                    justify-content: center;
                    margin: 0 auto 20px;
                    animation: afl-scaleIn 0.5s ease;
                }
                
                @keyframes afl-scaleIn {
                    0% { transform: scale(0); }
                    50% { transform: scale(1.2); }
                    100% { transform: scale(1); }
                }
                
                .afl-success-icon svg {
                    width: 40px;
                    height: 40px;
                    color: white;
                }
                
                .afl-progress {
                    display: flex;
                    justify-content: center;
                    gap: 8px;
                    margin-bottom: 20px;
                }
                
                .afl-progress-dot {
                    width: 8px;
                    height: 8px;
                    border-radius: 50%;
                    background: rgba(255,255,255,0.2);
                    transition: all 0.3s;
                }
                
                .afl-progress-dot.active {
                    background: ${this.config.color};
                    transform: scale(1.3);
                }
                
                @media (max-width: 480px) {
                    .afl-widget-btn span {
                        display: none;
                    }
                    .afl-widget-btn {
                        padding: 16px;
                        border-radius: 50%;
                    }
                    .afl-modal {
                        border-radius: 20px 20px 0 0;
                        max-height: 95vh;
                    }
                }
            `;

            const style = document.createElement('style');
            style.textContent = css;
            document.head.appendChild(style);
        },

        createButton: function() {
            const btn = document.createElement('button');
            btn.className = 'afl-widget-btn';
            btn.innerHTML = `
                <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                    <path d="M21 16v-2l-8-5V3.5c0-.83-.67-1.5-1.5-1.5S10 2.67 10 3.5V9l-8 5v2l8-2.5V19l-2 1.5V22l3.5-1 3.5 1v-1.5L13 19v-5.5l8 2.5z"/>
                </svg>
                <span>${this.t('button_text')}</span>
            `;
            btn.onclick = () => this.openModal();
            document.body.appendChild(btn);
        },

        createModal: function() {
            const overlay = document.createElement('div');
            overlay.className = 'afl-modal-overlay';
            overlay.id = 'afl-modal-overlay';
            overlay.onclick = (e) => { if (e.target === overlay) this.closeModal(); };

            overlay.innerHTML = `
                <div class="afl-modal">
                    <div class="afl-modal-header">
                        <button class="afl-modal-close" onclick="AFLCargoWidget.closeModal()">✕</button>
                        <h2 class="afl-modal-title">${this.t('title')}</h2>
                        <p class="afl-modal-subtitle">${this.t('subtitle')}</p>
                    </div>
                    <div class="afl-modal-body">
                        <div class="afl-progress">
                            <div class="afl-progress-dot active" data-step="1"></div>
                            <div class="afl-progress-dot" data-step="2"></div>
                            <div class="afl-progress-dot" data-step="3"></div>
                            <div class="afl-progress-dot" data-step="4"></div>
                        </div>
                        
                        <!-- Step 1: Transport -->
                        <div class="afl-step active" data-step="1">
                            <div class="afl-step-label">${this.t('step1')}</div>
                            <div class="afl-transport-grid">
                                <button class="afl-transport-btn" data-type="aereo" onclick="AFLCargoWidget.selectTransport('aereo')">
                                    <div class="icon">✈️</div>
                                    <div class="label">${this.t('air')}</div>
                                </button>
                                <button class="afl-transport-btn" data-type="maritimo" onclick="AFLCargoWidget.selectTransport('maritimo')">
                                    <div class="icon">🚢</div>
                                    <div class="label">${this.t('sea')}</div>
                                </button>
                                <button class="afl-transport-btn" data-type="terrestre" onclick="AFLCargoWidget.selectTransport('terrestre')">
                                    <div class="icon">🚛</div>
                                    <div class="label">${this.t('land')}</div>
                                </button>
                            </div>
                            <div class="afl-btn-row">
                                <button class="afl-btn afl-btn-primary" onclick="AFLCargoWidget.nextStep()" id="afl-next-1" disabled>→</button>
                            </div>
                        </div>
                        
                        <!-- Step 2: Route -->
                        <div class="afl-step" data-step="2">
                            <div class="afl-step-label">${this.t('step2')}</div>
                            <div class="afl-input-group">
                                <label class="afl-input-label">📍 ${this.t('origin')}</label>
                                <input type="text" class="afl-input" id="afl-origin" placeholder="Santiago, Chile">
                            </div>
                            <div class="afl-input-group">
                                <label class="afl-input-label">🎯 ${this.t('destination')}</label>
                                <input type="text" class="afl-input" id="afl-destination" placeholder="Miami, USA">
                            </div>
                            <div class="afl-btn-row">
                                <button class="afl-btn afl-btn-secondary" onclick="AFLCargoWidget.prevStep()">←</button>
                                <button class="afl-btn afl-btn-primary" onclick="AFLCargoWidget.nextStep()">→</button>
                            </div>
                        </div>
                        
                        <!-- Step 3: Weight -->
                        <div class="afl-step" data-step="3">
                            <div class="afl-step-label">${this.t('step3')}</div>
                            <div class="afl-weight-display">
                                <span class="afl-weight-value" id="afl-weight-display">100</span>
                                <span class="afl-weight-unit">kg</span>
                            </div>
                            <input type="range" class="afl-range" id="afl-weight" min="1" max="10000" value="100" oninput="AFLCargoWidget.updateWeight()">
                            <div class="afl-estimate" id="afl-estimate" style="display:none;">
                                <div class="afl-estimate-label">💰 Estimado</div>
                                <div class="afl-estimate-value" id="afl-estimate-value">$0 - $0 USD</div>
                            </div>
                            <div class="afl-btn-row">
                                <button class="afl-btn afl-btn-secondary" onclick="AFLCargoWidget.prevStep()">←</button>
                                <button class="afl-btn afl-btn-primary" onclick="AFLCargoWidget.nextStep()">→</button>
                            </div>
                        </div>
                        
                        <!-- Step 4: Contact -->
                        <div class="afl-step" data-step="4">
                            <div class="afl-step-label">${this.t('step4')}</div>
                            <div class="afl-input-group">
                                <label class="afl-input-label">${this.t('name')}</label>
                                <input type="text" class="afl-input" id="afl-name">
                            </div>
                            <div class="afl-input-group">
                                <label class="afl-input-label">${this.t('email')}</label>
                                <input type="email" class="afl-input" id="afl-email">
                            </div>
                            <div class="afl-input-group">
                                <label class="afl-input-label">${this.t('phone')}</label>
                                <input type="tel" class="afl-input" id="afl-phone">
                            </div>
                            <div class="afl-btn-row">
                                <button class="afl-btn afl-btn-secondary" onclick="AFLCargoWidget.prevStep()">←</button>
                                <button class="afl-btn afl-btn-primary" onclick="AFLCargoWidget.submit()" id="afl-submit">${this.t('submit')}</button>
                            </div>
                        </div>
                        
                        <!-- Success -->
                        <div class="afl-step" data-step="5">
                            <div class="afl-success">
                                <div class="afl-success-icon">
                                    <svg viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="3"><path d="M5 13l4 4L19 7"/></svg>
                                </div>
                                <h3 style="margin:0 0 8px">${this.t('success')}</h3>
                                <p style="color:rgba(255,255,255,0.6);margin:0 0 20px">${this.t('whatsapp')}</p>
                                <a href="#" id="afl-whatsapp-link" target="_blank" class="afl-btn afl-btn-whatsapp" style="text-decoration:none;display:inline-flex;">
                                    <svg width="20" height="20" fill="currentColor" viewBox="0 0 24 24"><path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347z"/></svg>
                                    WhatsApp
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            `;

            document.body.appendChild(overlay);
        },

        // State
        currentStep: 1,
        formData: {
            transport: '',
            origin: '',
            destination: '',
            weight: 100,
            name: '',
            email: '',
            phone: ''
        },

        openModal: function() {
            document.getElementById('afl-modal-overlay').classList.add('active');
            document.body.style.overflow = 'hidden';
        },

        closeModal: function() {
            document.getElementById('afl-modal-overlay').classList.remove('active');
            document.body.style.overflow = '';
        },

        selectTransport: function(type) {
            this.formData.transport = type;
            document.querySelectorAll('.afl-transport-btn').forEach(btn => {
                btn.classList.toggle('selected', btn.dataset.type === type);
            });
            document.getElementById('afl-next-1').disabled = false;
            this.calculateEstimate();
        },

        updateWeight: function() {
            const weight = document.getElementById('afl-weight').value;
            this.formData.weight = parseInt(weight);
            document.getElementById('afl-weight-display').textContent = parseInt(weight).toLocaleString();
            this.calculateEstimate();
        },

        calculateEstimate: function() {
            if (!this.formData.transport) return;
            
            const rates = { aereo: 3.5, maritimo: 0.15, terrestre: 0.8 };
            const mins = { aereo: 150, maritimo: 300, terrestre: 100 };
            
            let base = Math.max(this.formData.weight * rates[this.formData.transport], mins[this.formData.transport]);
            const min = Math.round(base * 0.8);
            const max = Math.round(base * 1.2);
            
            const el = document.getElementById('afl-estimate');
            if (el) {
                el.style.display = 'block';
                document.getElementById('afl-estimate-value').textContent = `$${min.toLocaleString()} - $${max.toLocaleString()} USD`;
            }
            
            this.formData.estimate = { min, max };
        },

        goToStep: function(step) {
            this.currentStep = step;
            document.querySelectorAll('.afl-step').forEach(el => {
                el.classList.toggle('active', parseInt(el.dataset.step) === step);
            });
            document.querySelectorAll('.afl-progress-dot').forEach(dot => {
                dot.classList.toggle('active', parseInt(dot.dataset.step) <= step);
            });
        },

        nextStep: function() {
            // Collect data from current step
            if (this.currentStep === 2) {
                this.formData.origin = document.getElementById('afl-origin').value;
                this.formData.destination = document.getElementById('afl-destination').value;
            }
            this.goToStep(this.currentStep + 1);
        },

        prevStep: function() {
            this.goToStep(this.currentStep - 1);
        },

        submit: function() {
            // Collect final data
            this.formData.name = document.getElementById('afl-name').value;
            this.formData.email = document.getElementById('afl-email').value;
            this.formData.phone = document.getElementById('afl-phone').value;

            const btn = document.getElementById('afl-submit');
            btn.textContent = this.t('sending');
            btn.disabled = true;

            // Simulate API call (replace with real endpoint)
            setTimeout(() => {
                // Update WhatsApp link
                const msg = `Hola! Solicite cotizacion:\n\n- Transporte: ${this.formData.transport}\n- Ruta: ${this.formData.origin} a ${this.formData.destination}\n- Peso: ${this.formData.weight} kg\n- Estimado: USD $${this.formData.estimate.min}-$${this.formData.estimate.max}\n\nSoy ${this.formData.name}`;
                document.getElementById('afl-whatsapp-link').href = `https://wa.me/${this.config.whatsapp}?text=${encodeURIComponent(msg)}`;
                
                this.goToStep(5);
            }, 1500);
        }
    };

    // Expose globally
    window.AFLCargoWidget = AFLCargoWidget;

    // Auto-init if data attributes present
    document.addEventListener('DOMContentLoaded', function() {
        const script = document.querySelector('script[data-afl-widget]');
        if (script) {
            AFLCargoWidget.init({
                whatsapp: script.dataset.whatsapp,
                color: script.dataset.color,
                position: script.dataset.position,
                lang: script.dataset.lang
            });
        }
    });
})();
